/********************************************************************
 * (C) Copyright 1998 by Hewlett-Packard GmbH. All rights reserved. *
 ********************************************************************/

#ifndef IO_H_INCLUDED
#define IO_H_INCLUDED

/* why were these 2 ever included? take it out!
#include <tdecdefs.h>
#include <propdefs.h>
*/ 

extern b_handlestype handle_array[];  /*lint !e762 ... redundant */

/* --------------------------------------------------------------------------
 * SCRobinson; 12 Nov 97.  This file declares functions and macros that
 * create an abstraction layer for performing I/O with the Best card.
 * -------------------------------------------------------------------------- */


/* --------------------------------------------------------------------------
 * Notes on Reading/Writing to/from the Best cards.
 *
 * With the introduction of the Best 2926, the I/O model of the C-API has 
 * been changed to create, as far as possible, single-point dependencies 
 * on the following variables:
 *
 * a. Best Model.
 * b. Protocols (Firmware knowledge, Byte Ordering etc.)
 * c. Operating System.
 * d. Ports (Drivers).
 *
 * Some concessions have been made to speed and the dependencies are not 100%
 * isolated.
 * 
 * a. Best Model Dependency:
 *    The firmware architecture of the 2926 is fundamentally different to
 *    the 2925.  With very few exceptions the firmware functions of the 2926 
 *    cannot be invoked using the same register-based algorithms as the 2925.
 *    There is no automatic translation from the 2925 registers to those of
 *    the 2926.  This means that each exported C-API function (and many internal
 *    functions) has 2 paths;
 *    one for 16-bit register sets and one for the older 8-bit register sets.
 *    The exported 8-bit register I/O interfaces to the cards remain the same; 
 *    BestBasicBlockRead() and BestBasicBlockWrite().  
 *    However, except in Offline mode and for registers below 0x0A,
 *    these calls CANNOT BE USED to communicate with a card using 16-bit regs.
 *
 * b. Protocol Dependency:
 *    Functions which are protocol aware are generally located in iocommon.c
 *    2925; Firmware knowledge is in BestBasicCommand8Bit().
 *    2926; Firmware knowledge is in BestBasicCommand().
 *
 *    Note that some port knowledge needed for consistent protocol translation
 *    is also  encapsulated in iocommon.c for ease of use.  
 *    Such functions include BestIsIoByteOrdered() and BestBasicNeedsAck().
 *
 *    For ALL I/O one of these functions is used;
 *
 *    BestBasicByteOrderRead(), BestBasicByteOrderWrite() 
 *      Have knowledge of endianess, can (and do) swap byte order as needed.
 *      Macros are provided (in b_io.h) for ease of use;
 *      - BestBasicWordRead(), BestBasicWordWrite()
 *      - BestBasicDWordRead(), BestBasicDWordWrite()
 *      
 *    BestBasicRead(), BestBasicWrite() (see c.)
 *      Have knowledge of Operating System AND Drivers, NO protocol knowledge.
 *      Macros are provided (in b_io.h) for ease of use;
 *      - BestBasicByteRead(), BestBasicByteWrite()
 *      - BestBasicByteStreamRead(), BestBasicByteStreamWrite()
 *
 * c. Operating System Dependency:
 *    I/O Functions which are NOT Driver but ARE operating system dependent are 
 *    located in ioxxx.c.  Note that this also gives us an opportunity to
 *    take advantage of OS driver interfaces if applicable.
 *    These functions are currently limited to:
 *    BestBasicRead(), BestBasicWrite().
 *
 * d. Port Dependency:
 *    Drivers (which are by definition dependent on the OS) are contained in
 *    files named with a combination of the port and the OS (i.e. pci32.c,
 *    pci16.c, ser32.c, sersicl.c etc.).
 *    Drivers provide only the most basic set of functions necessary to allow
 *    the higher level functions to operate correctly.  The following set of 
 *    functions are supplied by a typical driver:
 *
 *      - BestOpen___()
 *      - BestClose___()
 *
 *      - Best___RegWidthSet()
 *      - Best___DeviceConnect()
 *      - Best___CheckConnection()
 *      - Best___ReleaseConnection()
 *
 *    Not normally needed but available...
 *      - Best___OnReadError()
 *      - Best___OnWriteError()
 *
 *    These functions are not used in Win32 drivers because of the consistent 
 *    OS driver interface (ReadFile(), WriteFile()):
 *      - Best___BasicRead()
 *      - Best___BasicWrite()
 * -------------------------------------------------------------------------- */


/* --------------------------------------------------------------------------
 * Notes on "RegWidth".
 * The "register width" is always expressed in "number of bytes" but is used
 * in different places for completely different purposes;
 *
 * a. BestBasicByteOrderRead() and BestBasicByteOrderWrite().
 *    RegWidth is interpreted as the data-width (i.e. Int16, Int32 etc.) and
 *    is used for byte-ordering of the passed data.  The endianess of the
 *    host computer is also taken into account (big-endian hosts need no
 *    byte ordering).
 *    The data is assumed to be an array of 1 or more dataitems,
 *    each RegWidth bytes wide.
 *
 * b. BestBasicRead() and BestBasicWrite()
 *    RegWidth is ONLY used to make the call to BestRegwidthSet().
 *    The register width in this context could be more aptly called the
 *    "bus width to use" variable and determines how wide the data width
 *    of each read/write is that the final DRIVER uses for transferring the
 *    stream.
 *
 *    Drivers that are restricted to 8-bit buses (serial, parallel etc.)
 *    ignore RegWidth.  See BestIsIoByteOrdered() for additional comments.
 *
 *    We do not actually "pass" RegWidth to the driver.  Instead, we call
 *    BestRegwidthSet() which, as a minimum, stores the RegWidth in the handle
 *    array for retrieval by the driver.  
 *
 *    *** Kernel-mode drivers MUST have a separate IO-CTRL to set RegWidth. ***
 *    Anybody can call BestBasicRead/BestBasicWrite directly 
 *    ... BUT both functions call BestRegwidthSet(); 
 *    thus BestRegwidthSet() cannot use BestBasicWrite() to set the register 
 *    width or communicate with the driver.
 *
 * This mechanism has a number of consequences;
 *
 * 1. Reading or writing a BYTE stream currently forces ANY driver to restrict
 *    the bus width to 1 byte.  The byte-ordering data width WILL (and MUST)
 *    be the same as the bus width used.  Note that a big-endian host would
 *    not be bound by this restriction but drivers are not endianess-aware.
 * 2. If the driver is capable of reading/writing the native data-width of
 *    the passed array then NO BYTE-ORDERING NEEDS TO BE PERFORMED !!!
 *    Thus array I/O across the PCI bus will be the fastest.
 * 3. Side-stepping the routines provided (specifically BestBasicRead/Write)
 *    can have disastrous consequences because the bus-width is "remembered"
 *    and will be used by the driver until reset.  Thus every I/O MUST,
 *    at some point, go through BestBasicRead/BestBasicWrite or implement
 *    its own call to BestRegwidthSet().
 * 4. Asynchronous I/O is not feasible because we have only one RegWidth for
 *    both read and write.
 * -------------------------------------------------------------------------- */



/* --------------------------------------------------------------------------
 * Declarations in this file refer to functions in iocommon.c
 * -------------------------------------------------------------------------- */

/* --------------------------------------------------------------------------
 * Some useful macros
 * -------------------------------------------------------------------------- */

/* Note; you can define little-endianess as 1 to save time during I/O.
 * but BestIsHostLittleEndian() is quite fast and gives the correct 
 * answer at run-time.
 */
int BestIsCurMachineLittleEndian(void);   /* NEVER CALL THIS DIRECTLY */

/* #define BestIsHostLittleEndian   1 */
#define BestIsHostLittleEndian  BestIsCurMachineLittleEndian


/* --------------------------------------------------------------------------
 * Other Macros ... mostly for asserts.
 * -------------------------------------------------------------------------- */


/* TODO: move these makros to typedefs.h or delete them */
#ifdef BEST_DEBUG

# define BestIsHandleOpen(handle)   \
          (((handle) >= 0) && ((handle) < MAXHANDLES) && \
          (handle_array[(handle)].portnumber != INVALID_OS_HANDLE) && \
          (handle_array[(handle)].is_open != 0))

# define BestIsHandleConnected(handle)   \
          (((handle) >= 0) && ((handle) < MAXHANDLES) && \
          (handle_array[(handle)].portnumber != INVALID_OS_HANDLE) && \
          (handle_array[(handle)].is_open != 0) && \
          (handle_array[(handle)].is_connected != 0))

# define B_USE(var) (var)

#else

/* the run time "not so thorough" versions */
# define BestIsHandleOpen(handle)       \
          (((handle) >= 0) && ((handle) < MAXHANDLES) && \
          (handle_array[(handle)].is_open))

# define BestIsHandleConnected(handle) \
          (((handle) >= 0) && ((handle) < MAXHANDLES) && \
          (handle_array[(handle)].is_connected))


# define B_USE(var)

#endif  /* BEST_DEBUG */

#define BestIsRegWidthValid(RegWidth)   \
        ((1 == (RegWidth)) || (2 == (RegWidth)) || (4 == (RegWidth)))



/* TODO; remove dbg instances of this when all functions are 16-bit reg. aware */
#define IN_OFFLINE_MODE (B_PORT_OFFLINE == handle_array[handle].port)


/* --------------------------------------------------------------------------
 * these functions are for opening/closing a port connection with error and 
 * connection checks
 * -------------------------------------------------------------------------- */

b_errtype BestOpenConnection(b_handletype handle, b_int32 timeout);
void BestReleaseConnection(b_handletype handle);

b_errtype BestCheckConnection(b_handletype handle);
b_errtype BestIsDisconnected(b_handletype handle);

b_errtype BestCheckCallBack(b_handletype handle);

void BestCBReadStringSet (b_handletype handle, char *s);
void BestCBWriteStringSet (b_handletype handle, char *s);


b_errtype BestOnReadError(b_handletype handle);
b_errtype BestOnWriteError(b_handletype handle);


/* --------------------------------------------------------------------------
 * Direct Reading from the card....
 * -------------------------------------------------------------------------- */

/*********************** USE THESE MACROS *************************************/
/* defines for single data elements and byte streams */
#define BestBasicByteRead(handle,pData)  \
        BestBasicRead((handle),(pData),1UL,1)

#define BestBasicWordRead(handle,pData) \
        BestBasicByteOrderRead((handle),(pData),1UL,2)

#define BestBasicDWordRead(handle,pData) \
        BestBasicByteOrderRead((handle),(pData),1UL,4)

#define BestBasicByteStreamRead(handle,pData,NumBytes)  \
        BestBasicRead((handle),(pData),(NumBytes),1)


/*********************** DO NOT CALL THESE FUNCTIONS DIRECTLY *****************/
/* the "main man" for arrays */
b_errtype BestBasicByteOrderRead(
    b_handletype handle,
    b_int8ptr  pData,       /* caller's buffer */
    b_int32 NumDataElt,   /* number of data elements to read (NOT bytes) */
    b_int8 DataEltWidth); /* data element width in bytes */

/* stream reads ... NO byte ordering.  
 * Note that this function is OS-DEPENDENT... in io##.c (not iocommon.c)
 */
b_errtype BestBasicRead(
    b_handletype handle,
    b_int8ptr  pData,       /* caller's buffer */
    b_int32 NumBytes,     /* number of bytes to read */
    b_int8 DataEltWidth); /* data element width in bytes */


/* --------------------------------------------------------------------------
 * Direct Writing to the card....
 * -------------------------------------------------------------------------- */

/*********************** USE THESE MACROS *************************************/
/* defines for single data elements and byte streams */
#define BestBasicByteWrite(handle,pData) \
	BestBasicWrite((handle),(pData),1UL,1)

#define BestBasicWordWrite(handle,pData) \
	BestBasicByteOrderWrite((handle),(pData),1UL,2)

#define BestBasicDWordWrite(handle,pData) \
        BestBasicByteOrderWrite((handle),(pData),1UL,4)

#define BestBasicByteStreamWrite(handle,pData,NumBytes)  \
        BestBasicWrite((handle),(pData),(NumBytes),1)

/*********************** DO NOT CALL THESE FUNCTIONS DIRECTLY *****************/
/* the "main man" for arrays */
b_errtype BestBasicByteOrderWrite(
    b_handletype handle,
    b_int8ptr  pData,       /* caller's buffer */
    b_int32 NumDataElt,   /* number of data elements to write (NOT bytes) */
    b_int8 DataEltWidth); /* data element width in bytes */

/* stream writes ... NO byte ordering 
 * Note that this function is OS-DEPENDENT... in io##.c (not iocommon.c)
 */
b_errtype BestBasicWrite(
    b_handletype handle,
    b_int8ptr  pData,       /* caller's buffer */
    b_int32 NumBytes,     /* number of bytes to write */
    b_int8 DataEltWidth); /* data element width in bytes */

/* --------------------------------------------------------------------------
 * See the discussion on RegWidth above
 * -------------------------------------------------------------------------- */

b_errtype BestRegwidthSet(b_handletype handle, b_int8 regwidth);


/* --------------------------------------------------------------------------
 * Set timeouts....pass pCallersTimeouts = NULL to restore defaults 
 * -------------------------------------------------------------------------- */
#ifdef __cplusplus
extern "C" {
#endif


/* TODO; these default timeouts need confirmation */

/* Simple timeouts */
#define TIMEOUT_MAILBOX_PCI_READ  1000UL  /* milliseconds */
#define TIMEOUT_MAILBOX_PCI_WRITE 1000UL  /* milliseconds */


/* Full-featured timeouts */
/* Note that the read interval is important for response...
 * Serial set to about twice the norm for 9600 baud
 */

/* values have the following meaning
   - read interval
   - read total multiplier (should be set to slowest transfer time)
   - read total constant
   - write total multiplier
   - write total constant

   For additional help see MSVC help on COMMTIMEOUTS.
*/

#ifdef BEST_RELEASE
#define TIMEOUTS_DEF_SERIAL   {1000UL, 2UL, 10000UL, 2UL, 5000UL}
#define TIMEOUTS_E2925_SERIAL {1000UL, 1000UL, 15000UL, 2UL, 5000UL}
#define TIMEOUTS_DEF_PAR      {1000UL, 2UL, 10000UL, 2UL, 5000UL}
#define TIMEOUTS_DEF_PCI      {1000UL, 2UL, 10000UL, 2UL, 5000UL}
#define TIMEOUTS_DEF_FHIF     {1000UL, 2UL, 10000UL, 2UL, 5000UL}

#else
#define TIMEOUTS_DEF_SERIAL   {1000UL, 2UL, 10000UL, 2UL, 4500UL}
#define TIMEOUTS_E2925_SERIAL {1000UL, 1000UL, 15000UL, 2UL, 4500UL}
#define TIMEOUTS_DEF_PAR      {1000UL, 2UL, 10000UL, 2UL, 4500UL}
#define TIMEOUTS_DEF_PCI      {1000UL, 2UL, 10000UL, 2UL, 4500UL}
#define TIMEOUTS_DEF_FHIF     {1000UL, 2UL, 10000UL, 2UL, 4500UL}

#endif

#define TIMEOUTS_SET            0   /* set timeouts to values in passed struct */
#define TIMEOUTS_SET_DEF        1   /* alternate method of setting defaults */
#define TIMEOUTS_SET_NONE       2   /* set timeouts to infinite (none) */
#define TIMEOUTS_SET_IMM_READ   3   /* set read timeout immediate, write infinite */
#define TIMEOUTS_SET_IMM_WRITE  4   /* set write timeout immediate, read infinite */
#define TIMEOUTS_SET_IMM_IO     5   /* set read/write timeouts to immediate */

b_errtype EXPORT BestPortTimeoutSet(
                    b_handletype handle, 
                    BESTTIMEOUTS * pCallersTimeouts,
                    int fSetAction);


/* --------------------------------------------------------------------------
 * This section concerns writing directly to I/O ports.
 * The macros used are READPORT(portid) and WRITEPORT(portid, byte)
 * -------------------------------------------------------------------------- */

#define DIRECT_IO   0    /* the default */

/* we ONLY use direct io when compiling Win95 */

#if !defined(WINNT) && !defined(UNIX)

/* do nothing */

# if (defined(_WIN32) || defined(_WIN64)) && !defined(WIN95)

/* this is bad! */
#   error Unanticipated Win32 Operating System

# else

#   undef DIRECT_IO
#   define DIRECT_IO   1

/* these 2 functions are defined in io32.c */
void BestOutPortB(b_int16 portid, b_int8 uchar);
b_int8 BestInPortB(b_int16 portid);

# define READPORT  BestInPortB
# define WRITEPORT BestOutPortB
# define IOADDR(x,y) (b_int16)((b_int16)(x) + (b_int16)(y))
typedef b_int16 b_PortAddressType;

# endif  /* defined(_WIN32) && !defined(WIN95) */

#endif  /* !defined(WINNT) && !defined(UNIX) */

#ifdef __cplusplus
}
#endif

#endif
